(* ::Subsubsection:: *)
(*SchmidtCassegrainCBeforeSM package*)

(* ::Text:: *)
(*This impements the Schmidt Cassegrain telescope with a corrector in front of the Cassegrain combination and a spherical mirror*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/SchmidtCassegrainCBeforeSM"}]*)

BeginPackage["GeometricOptics`Packages`SchmidtCassegrainCBeforeSM`", {"GeometricOptics`"}]

Options[SchmidtCassegrainCBeforeSM] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
SchmidtCassegrainCBeforeSM::usage="SchmidtCassegrainCBeforeSM calculates parameters for a Schmidt Cassegrain telescope with a corrector in front of the Cassegrain combination.";

SchmidtCassegrainCBeforeSMInputPanel::usage = "SchmidtCassegrainCBeforeSMInputPanel gives the GUI panel to work with SchmidtCassegrainCBeforeSM";

$SchmidtCassegrainCBeforeSMInputVariables = {"f1", "ft", "em", "diam", "Ni", "\[CapitalDelta]", "\[Theta]"};

Begin["`Private`"]

SchmidtCassegrainCBeforeSM[{f1_, ft_, em_, diam_, Ni_, capitaldelta_, theta_}, opts___]:= SchmidtCassegrainCBeforeSM[f1, ft, em, diam, Ni, capitaldelta, theta, opts];

SchmidtCassegrainCBeforeSM[f1_, ft_, em_, diam_, Ni_, capitaldelta_, theta_, OptionsPattern[]] := 
 Module[{fact, \[Alpha], M, R2, D10, D11, D12, D20, D21, D22, D30, D31, D32, a41, a21, Rc, sol1, fun1, sag1, a42, K22, a22, Rc2,
		 fun2, sag2, a43, K13, K23, a23, Rc3, sol3, fun3, sag3, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["SchmidtCassegrainCBeforeSM", {$SchmidtCassegrainCBeforeSMInputVariables, {f1, ft, em, diam, Ni, capitaldelta, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 10, "PackageName" -> "SchmidtCassegrainCBeforeSM", 
					   "Arguments" ->  <|"f1" -> f1, "ft" -> ft, "em" -> em, "diam" -> diam, "Ni" -> Ni, "\[CapitalDelta]" -> capitaldelta, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
	
			fact = 0.866;
			\[Alpha] = em/f1;
			M = ft/f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = (2 M (1 + \[Alpha]))/(1 - M^2);
			R2 = \[Gamma] f1;
			
			D10 = -((1 + M^3 + \[Alpha] - M^2 (1 + \[Alpha]))/(32 f1^3 M^3));
			D11 = -(1/(32 f1^3));
			D12 = ((-1 + M)^3 (1 + \[Alpha]))/(32 f1^3 M^3 (1 + M));
			D20 = (M + \[Alpha] - M^3 (-1 + \[Delta]) - \[Delta] - \[Alpha] \[Delta] + M^2 (\[Alpha] (-1 + \[Delta]) + \[Delta]))/(8 f1^2 M^3);
			D21 = -(\[Delta]/(8 f1^2));
			D22 = ((-1 + M)^3 (M + \[Alpha] (-1 + \[Delta]) + \[Delta]))/(8 f1^2 M^3 (1 + M));
			D30 = 1/(8 f1 M^3 (1 + \[Alpha]))(M^4 + 2 M (\[Alpha] (-1 + \[Delta]) + \[Delta]) - 
												(\[Alpha] (-1 + \[Delta]) + \[Delta])^2 + 
												M^2 (1 + \[Alpha]) (-1 + \[Delta]) (1 + \[Alpha] (-1 + \[Delta]) + \[Delta]) - 
												M^3 (4 - 2 \[Delta] + \[Delta]^2 + \[Alpha] (2 - 2 \[Delta] + \[Delta]^2)));
			D31 = -(\[Delta]^2/(8 f1));
			D32 = ((-1 + M)^3 (M + \[Alpha] (-1 + \[Delta]) + \[Delta])^2)/(8 f1 M^3 (1 + M) (1 + \[Alpha]));
			
			a41 = -(D10/(-1 + Ni));
			a21 = -(1/2) a41 fact^2 diam^2;
			Rc = 1/(2 a21);
			\[Delta] = (M + M^3 + \[Alpha] - M^2 \[Alpha])/(1 - M^2 + M^3 + \[Alpha] - M^2 \[Alpha]);
			
			TotalAberrations[{10^9, -(1/(a4 (fact^2)(diam^2))), -2 f1, \[Gamma] f1}, {capitaldelta, \[Delta] f1, -\[Beta] f1}, 
								{{1, Ni, 1, -1, 1}}, {0, {a4, as}, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			
			sol1 = FindRoot[GOaberration[[1]] == 0, {a4, a41}];
			fun1 = -(1/2) ReplaceAll[a4, sol1] fact^2 diam^2 Global`y^2 + ReplaceAll[a4, sol1] Global`y^4;
			sag1 = ReplaceAll[fun1, {Global`y -> fact diam}];

			outputs = { {"Primary radius", -2 f1},
						{"Secondary radius", R2},
						{"Secondary diameter", (1 - \[Beta]) diam},
						{"Distance between primary and secondary", -\[Beta] f1},
						{"Obstruction coefficient", 1 - \[Beta]},
						{"Elimination of spherical aberration and coma with spherical mirrors", ""},
						{"Distance between corrector and primary mirror", \[Delta] f1},
						{"Aspheric constant of corrector", a41},						
						{"Equation of corrector", TraditionalForm[fun1]},
						{"Curvature radius of the corrector at the vertex", -(1/(ReplaceAll[a4, sol1] fact^2 diam^2))},
						{"Sagitta of corrector", sag1},
						{"Total spherical aberration", GOaberration[[1]] /. sol1},
						{"Total coma", GOcoma[[1]] /. sol1},
						{"Total astigmatism", GOastigmatism[[1]] /. sol1},
						{"Image heigth", GOimageHeight[[1, GOn]] /. sol1},
						{"Focal length", GOfocalLength[[1]] /. sol1}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"Back distance", "em", em},
							{"Refractive index of the corrector", "Ni", Ni},
							{"Thickness of the corrector", "thick", thick},
							{"Fraction of distance corrector-primary in focal unity", "delta", delta},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCassegrainCBeforeSM" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "SchmidtCassegrainCBeforeSM", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCassegrainCBeforeSM" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "SchmidtCassegrainCBeforeSM", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ft", ft},
														 {"em", em},
														 {"diam", diam},
														 {"Ni", Ni},
														 {"\[CapitalDelta]", capitaldelta},														 
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["SchmidtCassegrainCBeforeSM not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

SchmidtCassegrainCBeforeSMInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "SchmidtCassegrainCBeforeSM"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, diam, Ni, capitaldelta, theta} = ReplaceAll[$SchmidtCassegrainCBeforeSMInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Refractive index of the corrector", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Insert the refractive index of the corrector"]},
						  {"Distance between the corrector and the mirror", "\[CapitalDelta]", Tooltip[InputField[Dynamic[capitaldelta], Alignment -> Center], "Insert the distance between the corrector and the mirror"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, diam, Ni, capitaldelta, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, diam, Ni, capitaldelta, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]
  